<?php
session_start();
require_once __DIR__ . '/../includes/init.php';

// This is a public page - no login required
// Get maximum requests setting
$maxRequestsPerDay = (int)get_setting('maximum_amount_request', 5);

// Build API base URL so this file works both when included from /index.php
// and when opened directly as /public/certificate-search.php.
$scriptDir = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
$apiBase = preg_match('#/public$#', $scriptDir) ? ($scriptDir . '/api') : ($scriptDir . '/public/api');
?>
<!doctype html>
<html lang="de">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Zertifikatsabfrage</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Space+Grotesk:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Space Grotesk', sans-serif;
            font-size: 16px;
            line-height: 1.6;
            background-color: #f9f9f9;
            padding: 20px;
            color: #333;
        }

        /* iFrame-optimized: no full-page layout */
        .certificate-wizard {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            overflow: hidden;
        }

        /* Step containers */
        .step {
            padding: 40px 30px;
            display: none;
            animation: fadeIn 0.3s ease-in;
        }

        .step.active {
            display: block;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        /* Headings */
        h2 {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 20px;
            color: #1a1a1a;
        }

        /* Paragraphs */
        p {
            font-size: 16px;
            margin-bottom: 15px;
            color: #555;
            line-height: 1.6;
        }

        /* Form styling */
        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            font-size: 16px;
            font-weight: 500;
            margin-bottom: 8px;
            color: #333;
        }

        input[type="text"],
        input[type="email"],
        textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-family: 'Space Grotesk', sans-serif;
            font-size: 16px;
            transition: border-color 0.3s;
        }

        input[type="text"]:focus,
        input[type="email"]:focus,
        textarea:focus {
            outline: none;
            border-color: #366cd8;
            box-shadow: 0 0 0 3px rgba(54, 108, 216, 0.1);
        }

        textarea {
            resize: vertical;
            min-height: 100px;
        }

        /* Autocomplete dropdown */
        .autocomplete-list {
            border: 1px solid #ddd;
            border-top: none;
            max-height: 200px;
            overflow-y: auto;
            background: white;
            display: none;
            margin-top: -4px;
            border-radius: 0 0 4px 4px;
        }

        .autocomplete-list.active {
            display: block;
        }

        .autocomplete-item {
            padding: 12px;
            cursor: pointer;
            border-bottom: 1px solid #f0f0f0;
            transition: background-color 0.2s;
        }

        .autocomplete-item:last-child {
            border-bottom: none;
        }

        .autocomplete-item:hover {
            background-color: #f5f5f5;
        }

        .autocomplete-item.selected {
            background-color: #e8f0ff;
            color: #366cd8;
        }

        /* Loading spinner */
        .spinner {
            display: inline-block;
            margin-left: 8px;
        }

        .spinner i {
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        /* Buttons */
        button {
            width: 100%;
            padding: 12px 20px;
            background-color: #366cd8;
            color: white;
            border: none;
            border-radius: 4px;
            font-family: 'Space Grotesk', sans-serif;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.3s;
            margin-top: 10px;
        }

        button:hover {
            background-color: #2554b0;
        }

        button:disabled {
            background-color: #ccc;
            cursor: not-allowed;
        }

        /* Success/Error styling */
        .success-icon {
            font-size: 48px;
            color: #28a745;
            margin: 20px 0;
            text-align: center;
        }

        .error-icon {
            font-size: 48px;
            color: #dc3545;
            margin: 20px 0;
            text-align: center;
        }

        /* Confirmation text */
        .confirmation-box {
            background-color: #f8f9fa;
            border-left: 4px solid #366cd8;
            padding: 15px;
            margin: 20px 0;
            border-radius: 4px;
        }

        .confirmation-box p {
            margin: 5px 0;
        }

        .supplier-name {
            color: #1a1a1a;
            font-weight: 600;
            word-break: break-word;
        }

        .certificate-number {
            color: #1a1a1a;
            font-weight: 600;
        }

        .product-type {
            color: #1a1a1a;
            font-weight: 600;
        }

        .change-link {
            color: #366cd8;
            cursor: pointer;
            text-decoration: underline;
            font-weight: 500;
        }

        .change-link:hover {
            color: #2554b0;
        }

        /* Step counter */
        .step-counter {
            font-size: 14px;
            color: #999;
            margin-bottom: 30px;
            text-align: center;
        }

        /* Notice messages */
        .notice {
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
            font-size: 16px;
        }

        .notice.info {
            background-color: #d1ecf1;
            border-left: 4px solid #17a2b8;
            color: #0c5460;
        }

        .notice.error {
            background-color: #f8d7da;
            border-left: 4px solid #f5c6cb;
            color: #721c24;
        }

        .notice.success {
            background-color: #d4edda;
            border-left: 4px solid #c3e6cb;
            color: #155724;
        }
        .form_permanant_notice {
            position: relative;
            margin-top: 20px;
            opacity: 1;
            transition: opacity 0.3s ease;
        }
        .form_permanant_notice.fade-out {
            opacity: 0;
        }
        .notice-close-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            width: 28px;
            height: 28px;
            border: 1px solid #e2a8ad;
            border-radius: 50%;
            background: #fff;
            color: #721c24;
            font-size: 16px;
            font-weight: 700;
            line-height: 1;
            cursor: pointer;
            margin: 0;
            padding: 0;
        }
        .notice-close-btn:hover {
            background: #f8d7da;
        }

        /* Center text */
        .text-center {
            text-align: center;
        }

        /* Responsive */
        @media (max-width: 600px) {
            .step {
                padding: 30px 20px;
            }

            h2 {
                font-size: 20px;
            }

            p {
                font-size: 15px;
            }
        }
    </style>
</head>
<body>

<div class="certificate-wizard">
    
    <!-- STEP 1: Select Supplier -->
    <div id="step1" class="step active">
        <div class="step-counter">Schritt 1 von 3</div>
        <h2>Zertifikatsabfrage</h2>
        <p>Wir freuen uns, Sie dabei zu unterstützen Ihre Zertifikate zu validieren.</p>
        <p>Um den bestmöglichen Datenschutz für sowohl den Lieferanten, als auch den Endabnehmer, zu gewährleisten, haben wir eine Zertifikatsabfragemaske für Sie eingerichtet.</p>
        <p>Wir bitten um Verständnis, dass aus Datenschutzgründen nur <strong><?php echo $maxRequestsPerDay; ?></strong> Anfragen pro Tag zugelassen sind. Vielen Dank.</p>
        
        <div class="form-group">
            <label for="search_name">Lieferantenname</label>
            <input 
                type="text" 
                id="search_name" 
                name="search_name" 
                placeholder="Bitte geben Sie den Namen Ihres Lieferanten an"
                autocomplete="off"
            >
            <div id="autocomplete_list" class="autocomplete-list"></div>
        </div>

        <button type="button" id="btn_next_step1" onclick="proceedToStep2()" disabled>
            Weiter <i class="fas fa-arrow-right" style="margin-left: 8px;"></i>
        </button>
    </div>

    <!-- STEP 2: Enter Certificate Number -->
    <div id="step2" class="step">
        <div class="step-counter">Schritt 2 von 3</div>
        <h2>Zertifikatsabfrage</h2>
        
        <div class="confirmation-box">
            <p>Sie haben <span class="supplier-name" id="supplier_display">-</span> als Ihren Lieferanten ausgewählt. <a class="change-link" onclick="goBackToStep1()">Lieferant ändern</a></p>
        </div>

        <p>Bitte geben Sie die Zertifikatsnummer an, welche der Lieferant Ihnen mitgeteilt hat.</p>

        <div class="form-group">
            <label for="search_certificate">Zertifikatsnummer</label>
            <input 
                type="text" 
                id="search_certificate" 
                name="search_certificate" 
                placeholder="Bitte geben Sie die Zertifizierungsnummer ein"
            >
        </div>

        <button type="button" id="btn_search_cert" onclick="searchCertificate()">
            Anfrage starten <i id="loader_cert" class="fas fa-spinner fa-spin" style="display:none; margin-left: 8px;"></i>
        </button>
    </div>

    <!-- STEP 3A: Success Result -->
    <div id="step3_success" class="step">
        <div class="step-counter">Schritt 3 von 3</div>
        <h2 class="text-center">Zertifikatsabfrage</h2>
        <p class="text-center">Vielen Dank für Ihre Abfrage.</p>
        <div class="success-icon">
            <i class="fas fa-check-circle"></i>
        </div>
        <p class="text-center">Wir freuen uns Ihnen mitzuteilen, dass <span class="supplier-name" id="success_supplier">-</span> der rechtmäßige Inhaber des Zertifikats <span class="certificate-number" id="success_certificate">-</span> ist.</p>
        <p class="text-center">Das Zertifikat mit der Nummer <span class="certificate-number" id="success_cert_num">-</span> ist der Baumart <span class="product-type" id="success_product">-</span> zugeordnet.</p>
        <button type="button" onclick="restartForm()">Neue Anfrage starten</button>
    </div>

    <!-- STEP 3B: Failed Result -->
    <div id="step3_failed" class="step">
        <div class="step-counter">Schritt 3 von 3</div>
        <h2 class="text-center">Zertifikatsabfrage</h2>
        <p class="text-center">Vielen Dank für Ihre Abfrage.</p>
        <div class="error-icon">
            <i class="fas fa-times-circle"></i>
        </div>
        <p class="text-center">Es tut uns leid.<br>Die Zertifikatsnummer <span class="certificate-number" id="failed_cert_num">-</span> passt nicht zum Lieferanten <span class="supplier-name" id="failed_supplier">-</span></p>
        <p class="text-center">Bitte probieren Sie es noch einmal nach 14 Tagen (siehe Erläuterungen). Sollte sich die Meldung auf diese und/oder weitere Anfragen wiederholen, senden Sie uns gerne eine Nachricht. Wir setzen uns dann individuell mit dem Vorgang auseinander.</p>
        <button type="button" onclick="openManualInvestigation()">Manuelle Überprüfung anfragen</button>
        <button type="button" onclick="restartForm()" style="background-color: #6c757d; margin-top: 0;">Neue Anfrage starten</button>
    </div>

    <!-- STEP 4: Rate Limit Exceeded -->
    <div id="step_rate_limited" class="step">
        <div class="step-counter">Abfrage nicht möglich</div>
        <h2 class="text-center">Zertifikatsabfrage</h2>
        <div class="error-icon">
            <i class="fas fa-exclamation-circle"></i>
        </div>
        <div class="notice error">
            <p class="text-center"><strong>Tagesgrenze erreicht</strong></p>
            <p class="text-center">Sie haben das Abfragelimit von <?php echo $maxRequestsPerDay; ?> Anfragen pro Tag erreicht. Bitte versuchen Sie morgen erneut.</p>
        </div>
        <button type="button" onclick="restartForm()" style="background-color: #6c757d;">Neue Anfrage starten</button>
    </div>

    <!-- Manual Investigation Form (shown in modal/overlay) -->
    <div id="manual_form_step" class="step">
        <div class="step-counter">Manuelle Überprüfung</div>
        <h2>Manuelle Überprüfung anfragen</h2>
        <p>Bitte füllen Sie das Formular aus, damit wir Ihre Anfrage überprüfen können.</p>

        <div class="notice info">
            <p><strong>Lieferant:</strong> <span class="supplier-name" id="manual_supplier_display">-</span></p>
            <p><strong>Zertifikatsnummer:</strong> <span class="certificate-number" id="manual_cert_display">-</span></p>
        </div>

        <div class="form-group">
            <label for="manual_your_name">Ihr Name *</label>
            <input type="text" id="manual_your_name" name="your_name" placeholder="Bitte geben Sie Ihren Namen an" required>
        </div>

        <div class="form-group">
            <label for="manual_your_email">Ihre E-Mail-Adresse *</label>
            <input type="email" id="manual_your_email" name="your_email" placeholder="Bitte geben Sie Ihre E-Mail-Adresse an" required>
        </div>

        <div class="form-group">
            <label for="manual_your_note">Nachricht (optional)</label>
            <textarea id="manual_your_note" name="your_note" placeholder="Schreiben Sie hier weitere Informationen, falls erforderlich..."></textarea>
        </div>

        <button type="button" id="btn_submit_manual" onclick="submitManualInvestigation()">
            Abfrage absenden <i id="loader_manual" class="fas fa-spinner fa-spin" style="display:none; margin-left: 8px;"></i>
        </button>

        <button type="button" onclick="goBackToStep1()" style="background-color: #6c757d; margin-top: 0;">Abbrechen</button>
    </div>

    <!-- Success: Manual Form Submitted -->
    <div id="manual_success_step" class="step">
        <h2 class="text-center">Vielen Dank</h2>
        <div class="success-icon">
            <i class="fas fa-check-circle"></i>
        </div>
        <p class="text-center">Ihre Anfrage für eine manuelle Überprüfung wurde erfolgreich eingereicht. Wir werden Ihre Anfrage überprüfen und uns mit Ihnen in Verbindung setzen.</p>
        <button type="button" onclick="restartForm()">Neue Anfrage starten</button>
    </div>

    <div id="form_permanant_notice" class="form_permanant_notice">
        <div class="notice error">
            <button type="button" class="notice-close-btn" aria-label="Close notice" onclick="closePermanentNotice()">x</button>
            <p class="text-center"><strong>Wichtige Erläuterung</strong></p>
            <p class="text-center">1. Bitte verwenden Sie KEINE Umlaute - nehmen Sie "ae", "oe" und "ue" sowie anstelle "ß" ein "ss". 2. Verwenden SIe ausschließlich die Nummer des FfV-Zertifikates. Die Stammzertifikatnummer oder Registernummer erbringt KEIN Ergebnis. 3. Die Zertifikatsabfrage befindet sich zurzeit in der Entwicklung. Bitte beachten Sie, dass eine solche Datenbank niemals tagesaktuell sein kann. Dies betrifft jedoch hier nur die Pflanzenzertifikate, diese erkennen Sie daran, dass das letzte Zeichen der Zertifikatsnummer ein p oder b ist. Pflanzenzertifikate werden beim Handel zwischen Baumschulen ausgestellt, das geschieht manchmal innerhalb weniger Tage, d.h. die Pflanzenlieferung erfolgt schneller als die Aktualisierung der Datenbank. Wenn Sie also bei der Abfrage die Meldung, „Es tut uns leid. Die Zertifikatsnummer ### passt nicht zum Lieferanten XY“ erhalten, probieren Sie es bitte nach 14 Tagen erneut. Selbstverständlich können Sie uns auch wie in der Vergangenheit Lieferscheine zukommen lassen, welche wir gerne für Sie prüfen, oder unser Labor zwecks Auskunft anrufen – 0551-3937100 oder 0172-2147195.</p>
        </div>
    </div>

</div>

<script>
// State variables
let selectedSupplier = null;
let selectedCertificate = null;
let maxRequests = <?php echo $maxRequestsPerDay; ?>;
const API_BASE = <?php echo json_encode($apiBase); ?>;

// Helper: Show step
function showStep(stepId) {
    document.querySelectorAll('.step').forEach(step => step.classList.remove('active'));
    const step = document.getElementById(stepId);
    if (step) {
        step.classList.add('active');
        // Scroll to top of form
        document.querySelector('.certificate-wizard').scrollIntoView({ behavior: 'smooth' });
    }
}

// STEP 1: Autocomplete supplier search
document.getElementById('search_name').addEventListener('input', function() {
    const query = this.value.trim();
    const btn = document.getElementById('btn_next_step1');
    const list = document.getElementById('autocomplete_list');

    if (query.length < 2) {
        list.classList.remove('active');
        selectedSupplier = null;
        btn.disabled = true;
        return;
    }

    // Fetch suppliers via AJAX
    fetch(API_BASE + '/search-suppliers.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'q=' + encodeURIComponent(query)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.suppliers.length > 0) {
            list.innerHTML = '';
            data.suppliers.forEach(supplier => {
                const item = document.createElement('div');
                item.className = 'autocomplete-item';
                item.textContent = supplier;
                item.onclick = () => selectSupplier(supplier);
                list.appendChild(item);
            });
            list.classList.add('active');
        } else {
            list.innerHTML = '<div class="autocomplete-item">Keine Einträge gefunden.</div>';
            list.classList.add('active');
            selectedSupplier = null;
            btn.disabled = true;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        list.classList.remove('active');
        btn.disabled = true;
    });
});

// Select supplier from autocomplete
function selectSupplier(name) {
    selectedSupplier = name;
    document.getElementById('search_name').value = name;
    document.getElementById('autocomplete_list').classList.remove('active');
    document.getElementById('btn_next_step1').disabled = false;
}

// Proceed to Step 2
function proceedToStep2() {
    if (!selectedSupplier) return;
    document.getElementById('supplier_display').textContent = selectedSupplier;
    document.getElementById('search_certificate').value = '';
    showStep('step2');
    document.getElementById('search_certificate').focus();
}

// Search certificate
function searchCertificate() {
    const certNum = document.getElementById('search_certificate').value.trim();
    if (!certNum) {
        alert('Bitte geben Sie eine Zertifikatsnummer ein.');
        return;
    }

    selectedCertificate = certNum;
    const loader = document.getElementById('loader_cert');
    const btn = document.getElementById('btn_search_cert');
    
    loader.style.display = 'inline-block';
    btn.disabled = true;

    // Fetch verification via AJAX
    fetch(API_BASE + '/verify-certificate.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'supplier=' + encodeURIComponent(selectedSupplier) + '&certificate=' + encodeURIComponent(certNum)
    })
    .then(response => response.json())
    .then(data => {
        loader.style.display = 'none';
        btn.disabled = false;

        if (data.status === 'success') {
            // Certificate found
            document.getElementById('success_supplier').textContent = selectedSupplier;
            document.getElementById('success_certificate').textContent = certNum;
            document.getElementById('success_cert_num').textContent = certNum;
            document.getElementById('success_product').textContent = data.product;
            showStep('step3_success');
        } else if (data.status === 'not_found') {
            // Certificate not found
            document.getElementById('failed_supplier').textContent = selectedSupplier;
            document.getElementById('failed_cert_num').textContent = certNum;
            showStep('step3_failed');
        } else if (data.status === 'rate_limited') {
            // Rate limit exceeded
            showStep('step_rate_limited');
        } else {
            alert('Ein Fehler ist aufgetreten: ' + (data.message || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        loader.style.display = 'none';
        btn.disabled = false;
        alert('Fehler bei der Verbindung zum Server.');
    });
}

// Open manual investigation form
function openManualInvestigation() {
    document.getElementById('manual_supplier_display').textContent = selectedSupplier;
    document.getElementById('manual_cert_display').textContent = selectedCertificate;
    showStep('manual_form_step');
}

// Submit manual investigation form
function submitManualInvestigation() {
    const name = document.getElementById('manual_your_name').value.trim();
    const email = document.getElementById('manual_your_email').value.trim();
    const note = document.getElementById('manual_your_note').value.trim();

    if (!name || !email) {
        alert('Bitte füllen Sie alle erforderlichen Felder aus.');
        return;
    }

    const loader = document.getElementById('loader_manual');
    const btn = document.getElementById('btn_submit_manual');
    
    loader.style.display = 'inline-block';
    btn.disabled = true;

    fetch(API_BASE + '/submit-investigation.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'supplier=' + encodeURIComponent(selectedSupplier) + 
              '&certificate=' + encodeURIComponent(selectedCertificate) +
              '&name=' + encodeURIComponent(name) +
              '&email=' + encodeURIComponent(email) +
              '&note=' + encodeURIComponent(note)
    })
    .then(response => response.json())
    .then(data => {
        loader.style.display = 'none';
        btn.disabled = false;

        if (data.success) {
            showStep('manual_success_step');
        } else {
            alert('Fehler beim Absenden der Anfrage: ' + (data.message || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        loader.style.display = 'none';
        btn.disabled = false;
        alert('Fehler bei der Verbindung zum Server.');
    });
}

// Go back to Step 1
function goBackToStep1() {
    selectedSupplier = null;
    selectedCertificate = null;
    document.getElementById('search_name').value = '';
    document.getElementById('search_certificate').value = '';
    document.getElementById('btn_next_step1').disabled = true;
    document.getElementById('autocomplete_list').classList.remove('active');
    showStep('step1');
    document.getElementById('search_name').focus();
}

// Restart form
function restartForm() {
    goBackToStep1();
}

function closePermanentNotice() {
    const notice = document.getElementById('form_permanant_notice');
    if (!notice) return;
    notice.classList.add('fade-out');
    setTimeout(() => {
        notice.style.display = 'none';
    }, 300);
}
</script>

</body>
</html>

