<?php
header('Content-Type: application/json');
require_once __DIR__ . '/../../includes/init.php';

function get_session_request_count_last_24h() {
    if (!isset($_SESSION['verify_request_timestamps']) || !is_array($_SESSION['verify_request_timestamps'])) {
        $_SESSION['verify_request_timestamps'] = [];
    }

    $cutoff = time() - 86400;
    $_SESSION['verify_request_timestamps'] = array_values(array_filter(
        $_SESSION['verify_request_timestamps'],
        function ($ts) use ($cutoff) {
            return (int)$ts >= $cutoff;
        }
    ));

    return count($_SESSION['verify_request_timestamps']);
}

function register_session_request_now() {
    if (!isset($_SESSION['verify_request_timestamps']) || !is_array($_SESSION['verify_request_timestamps'])) {
        $_SESSION['verify_request_timestamps'] = [];
    }
    $_SESSION['verify_request_timestamps'][] = time();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

$supplier = isset($_POST['supplier']) ? trim($_POST['supplier']) : '';
$certificate = isset($_POST['certificate']) ? trim($_POST['certificate']) : '';

if (empty($supplier) || empty($certificate)) {
    echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
    exit;
}

try {
    $db = getDB();
    $maxRequests = (int)get_setting('maximum_amount_request', 5);
    
    // Trigger automatic cron job (runs every 14 days if needed)
    run_automatic_email_cron();
    
    // Check rate limit per browser session in last 24 hours
    $requestCount = get_session_request_count_last_24h();
    
    if ($requestCount >= $maxRequests) {
        echo json_encode(['status' => 'rate_limited']);
        exit;
    }

    // Count this request for the current session.
    register_session_request_now();
    
    // Check if certificate exists for this supplier
    $stmt = $db->prepare("
        SELECT product 
        FROM " . table('WPAS_csv') . "
        WHERE username = ? AND certificate = ? 
        LIMIT 1
    ");
    $stmt->execute([$supplier, $certificate]);
    $certResult = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Record this search
    $resultStatus = $certResult ? 1 : 0;
    $stmt = $db->prepare("
        INSERT INTO " . table('WPAS_search_record') . " (username, certificate, result) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$supplier, $certificate, $resultStatus]);
    
    if ($certResult) {
        // Certificate found
        echo json_encode([
            'status' => 'success',
            'product' => $certResult['product']
        ]);
    } else {
        // Certificate not found
        echo json_encode(['status' => 'not_found']);
    }
    
} catch (Exception $e) {
    error_log('Certificate verification error: ' . $e->getMessage());
    echo json_encode(['status' => 'error', 'message' => 'Database error']);
}
