<?php
// Web installer: if app already installed, redirect to admin login.
// Otherwise show an installation form to collect DB and initial admin credentials.

ini_set('display_errors', 1);
error_reporting(E_ALL);

// Try to load existing config and verify installation
$installed = false;
$configPath = __DIR__ . '/includes/config.php';
if (file_exists($configPath)) {
	$cfg = include $configPath;
	try {
		$dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', $cfg['db_host'], $cfg['db_name']);
		$db = new PDO($dsn, $cfg['db_user'], $cfg['db_pass']);
		$db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		$row = $db->query('SELECT COUNT(*) AS c FROM users')->fetch(PDO::FETCH_ASSOC);
		if ($row && intval($row['c'] ?? 0) > 0) {
			$installed = true;
		}
	} catch (Exception $e) {
		$installed = false;
	}
}

if ($installed) {
	// Load certificate search form directly on home page (no redirect)
	include __DIR__ . '/public/certificate-search.php';
	exit;
}

$error = '';
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
	$db_host = trim($_POST['db_host'] ?? '127.0.0.1');
	$db_name = trim($_POST['db_name'] ?? '');
	$db_user = trim($_POST['db_user'] ?? 'root');
	$db_pass = trim($_POST['db_pass'] ?? '');
	$admin_email = trim($_POST['admin_email'] ?? '');
	$admin_pass = $_POST['admin_pass'] ?? '';
	$admin_pass2 = $_POST['admin_pass2'] ?? '';

	if (!$db_name || !$admin_email || !$admin_pass) {
		$error = 'Please fill required fields.';
	} elseif ($admin_pass !== $admin_pass2) {
		$error = 'Passwords do not match.';
	} elseif (strlen($admin_pass) < 6) {
		$error = 'Password must be at least 6 characters.';
	} else {
		try {
			// Connect to MySQL server (without db) to create database if needed
			$dsnServer = sprintf('mysql:host=%s;charset=utf8mb4', $db_host);
			$pdo = new PDO($dsnServer, $db_user, $db_pass, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);

			// Create database if not exists
			$pdo->exec("CREATE DATABASE IF NOT EXISTS `" . str_replace('`','``',$db_name) . "` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

			// Connect to the created database
			$dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', $db_host, $db_name);
			$dbConn = new PDO($dsn, $db_user, $db_pass, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);

			// Create users table
			$sql = "CREATE TABLE IF NOT EXISTS users (
				id INT AUTO_INCREMENT PRIMARY KEY,
				email VARCHAR(255) NOT NULL UNIQUE,
				password_hash VARCHAR(255) NOT NULL,
				reset_token VARCHAR(255) DEFAULT NULL,
				reset_expires INT DEFAULT NULL,
				pending_email VARCHAR(255) DEFAULT NULL,
				email_token VARCHAR(255) DEFAULT NULL,
				email_expires INT DEFAULT NULL,
				created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
			$dbConn->exec($sql);

			// --- Plugin tables: create schema and import data from mysql-dump/*.sql if present
			$dumpDir = __DIR__ . '/mysql-dump';
			$detectedPrefix = null;
			if (is_dir($dumpDir)) {
				$files = glob($dumpDir . '/*.sql');
				foreach ($files as $file) {
					$content = file_get_contents($file);
					// try to detect table name from CREATE TABLE or INSERT INTO
					if (preg_match('/CREATE TABLE `([^`]+WPAS_[^`]+)`/i', $content, $m) || preg_match('/INSERT INTO `([^`]+WPAS_[^`]+)`/i', $content, $m)) {
						$tableName = $m[1];
						// detect prefix (part before WPAS_)
						if ($detectedPrefix === null && preg_match('/^(.*?)WPAS_/i', $tableName, $p)) {
							$detectedPrefix = $p[1] ?: '';
						}

						// Determine table type and create table with expected schema
						if (stripos($tableName, 'WPAS_csv`') !== false || stripos($tableName, 'WPAS_csv') !== false) {
							$create = "CREATE TABLE IF NOT EXISTS `{$tableName}` (
								id BIGINT(20) NOT NULL AUTO_INCREMENT,
								username VARCHAR(220) DEFAULT NULL,
								certificate VARCHAR(220) DEFAULT NULL,
								product VARCHAR(220) DEFAULT NULL,
								date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
								PRIMARY KEY(id)
							) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
						} elseif (stripos($tableName, 'WPAS_search_record') !== false) {
							$create = "CREATE TABLE IF NOT EXISTS `{$tableName}` (
								id BIGINT(20) NOT NULL AUTO_INCREMENT,
								username VARCHAR(220) DEFAULT NULL,
								certificate VARCHAR(220) DEFAULT NULL,
								result VARCHAR(220) DEFAULT NULL,
								date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
								PRIMARY KEY(id)
							) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
						} elseif (stripos($tableName, 'WPAS_email_reports') !== false) {
							$create = "CREATE TABLE IF NOT EXISTS `{$tableName}` (
								id BIGINT(20) NOT NULL AUTO_INCREMENT,
								total_amount_requests VARCHAR(220) DEFAULT NULL,
								type VARCHAR(220) DEFAULT NULL,
								date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
								PRIMARY KEY(id)
							) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
						} else {
							$create = null;
						}

						if ($create) {
							// Drop existing and recreate to ensure expected schema
							try {
								$dbConn->exec("DROP TABLE IF EXISTS `{$tableName}`");
							} catch (Exception $e) {
								// ignore
							}
							$dbConn->exec($create);
						}

						// Extract and run INSERT statements only
						if (preg_match_all('/INSERT INTO `'.preg_quote($tableName,'/').'`.*?;/is', $content, $ins)) {
							foreach ($ins[0] as $insertSql) {
								try {
									// GDPR: never import historical IP-based search records.
									if (stripos($tableName, 'WPAS_search_record') !== false) {
										continue;
									}
									$dbConn->exec($insertSql);
								} catch (Exception $e) {
									// try to clean up backticks/charset related parts and continue
									// skip problematic inserts
								}
							}
						}
					}
				}
			}

			// Ensure WPAS_csv_record exists (not included in dumps)
			$prefix = $detectedPrefix ?? 'wp_';
			$csvRecordTable = $prefix . 'WPAS_csv_record';
			$dbConn->exec("CREATE TABLE IF NOT EXISTS `{$csvRecordTable}` (
				id BIGINT(20) NOT NULL AUTO_INCREMENT,
				total_entries VARCHAR(220) DEFAULT NULL,
				date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
				PRIMARY KEY(id)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

			// Create settings table for storing random app settings
			$sqlSettings = "CREATE TABLE IF NOT EXISTS settings (
				id INT AUTO_INCREMENT PRIMARY KEY,
				setting_key VARCHAR(255) NOT NULL UNIQUE,
				setting_value LONGTEXT,
				updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
			$dbConn->exec($sqlSettings);

			// Insert initial admin user (email as identifier)
			$hash = password_hash($admin_pass, PASSWORD_DEFAULT);
			$ins = $dbConn->prepare('INSERT INTO users (email, password_hash) VALUES (?, ?)');
			$ins->execute([$admin_email, $hash]);

			// Write includes/config.php
			$cfgArray = [
				'db_host' => $db_host,
				'db_name' => $db_name,
				'db_user' => $db_user,
				'db_pass' => $db_pass,
				'smtp' => [
					'host' => '',
					'port' => 587,
					'user' => '',
					'pass' => '',
					'secure' => 'tls',
					'from_email' => 'no-reply@example.com',
					'from_name' => 'My Plugin'
				]
			];
			$cfgContent = "<?php\n// Auto-generated by web installer.\nreturn ";
			$cfgContent .= var_export($cfgArray, true) . ";\n";
			$cfgFile = __DIR__ . '/includes/config.php';
			if (file_put_contents($cfgFile, $cfgContent) === false) {
				$error = 'Failed to write includes/config.php — check file permissions.';
			} else {
				$success = 'Installation complete. Redirecting to login...';
				header('Refresh:2; url=admin/login.php');
			}
		} catch (Exception $e) {
			$error = 'Installation error: ' . $e->getMessage();
		}
	}
}
?>
<!doctype html>
<html lang="en">
<head>
	<meta charset="utf-8">
	<meta name="viewport" content="width=device-width,initial-scale=1">
	<title>Install Application</title>
	<link rel="stylesheet" href="assets/css/style.css">
</head>
<body class="login-page">
<div class="login-wrap">
	<h1 class="wp-brand">Application Installer</h1>
	<?php if($error): ?><div class="notice error"><?php echo htmlspecialchars($error); ?></div><?php endif; ?>
	<?php if($success): ?><div class="notice success"><?php echo htmlspecialchars($success); ?></div><?php endif; ?>
	<?php if(!$success): ?>
	<form method="post">
		<h3>Database</h3>
		<label>DB Host
			<input type="text" name="db_host" value="127.0.0.1" required>
		</label>
		<label>DB Name
			<input type="text" name="db_name" value="ffv_zertifikat" required>
		</label>
		<label>DB User
			<input type="text" name="db_user" value="root" required>
		</label>
		<label>DB Password
			<input type="password" name="db_pass">
		</label>

		<h3>Admin Account</h3>
		<label>Email (will be used as username)
			<input type="email" name="admin_email" required>
		</label>
		<label>Password
			<input type="password" name="admin_pass" required>
		</label>
		<label>Confirm password
			<input type="password" name="admin_pass2" required>
		</label>

		<div class="form-actions">
			<button class="btn" type="submit">Install</button>
		</div>
	</form>
	<?php endif; ?>
</div>
</body>
</html>
