<?php
// Only start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
define('APP_ROOT', __DIR__ . '/..');

$config = include __DIR__ . '/config.php';

function getDB() {
    static $db = null;
    global $config;
    if ($db === null) {
        $dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', $config['db_host'], $config['db_name']);
        $db = new PDO($dsn, $config['db_user'], $config['db_pass']);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }
    return $db;
}

/* Table name helper - returns prefixed table names for consistency */
define('DB_PREFIX', 'wp_');
function detect_table_prefix() {
    static $prefix = null;
    if ($prefix !== null) {
        return $prefix;
    }

    // Fallback prefix if no WPAS tables are found.
    $prefix = DB_PREFIX;

    try {
        $db = getDB();
        $tables = $db->query('SHOW TABLES')->fetchAll(PDO::FETCH_COLUMN, 0);
        foreach ($tables as $tableName) {
            if (preg_match('/^(.*)WPAS_csv$/i', $tableName, $match)) {
                $prefix = $match[1];
                break;
            }
        }
    } catch (Exception $e) {
        // Keep fallback prefix if detection fails.
    }

    return $prefix;
}
function table($name) {
    return detect_table_prefix() . $name;
}

function get_admin_menu_items() {
    return [
        'dashboard' => ['label' => 'Dashboard', 'href' => 'dashboard.php'],
        'csv_import' => ['label' => 'CSV Importieren', 'href' => 'csv-import.php'],
        'suppliers' => ['label' => 'Lieferanten', 'href' => 'suppliers.php'],
        'search_records' => ['label' => 'Abfrage-Protokoll', 'href' => 'search-records.php'],
        'email_reports' => ['label' => 'Email Reports', 'href' => 'email-reports.php'],
        'settings' => ['label' => 'Einstellungen', 'href' => 'settings.php'],
    ];
}

function render_admin_sidebar($activeKey = '') {
    $items = get_admin_menu_items();
    echo '<aside class="sidebar"><nav><ul>';
    foreach ($items as $key => $item) {
        $activeClass = ($key === $activeKey) ? ' class="active"' : '';
        echo '<li' . $activeClass . '><a href="' . htmlspecialchars($item['href'], ENT_QUOTES, 'UTF-8') . '">'
            . htmlspecialchars($item['label'], ENT_QUOTES, 'UTF-8') . '</a></li>';
    }
    echo '</ul></nav></aside>';
}

function get_admin_brand_name() {
    return 'FFV Zertifikat';
}

function render_admin_topbar($userEmail, $logoutUrl = 'dashboard.php?action=logout') {
    $brand = htmlspecialchars(get_admin_brand_name(), ENT_QUOTES, 'UTF-8');
    $email = htmlspecialchars((string)$userEmail, ENT_QUOTES, 'UTF-8');
    $logout = htmlspecialchars($logoutUrl, ENT_QUOTES, 'UTF-8');

    echo '<header class="topbar"><div class="container">';
    echo '<div class="branding">' . $brand . '</div>';
    echo '<div class="top-actions"><span class="user">Hallo, ' . $email . '</span>';
    echo '<a href="' . $logout . '" class="btn small">Abmelden</a></div>';
    echo '</div></header>';
}

function render_admin_footer() {
    $brand = htmlspecialchars(get_admin_brand_name(), ENT_QUOTES, 'UTF-8');
    echo '<footer class="footer"><div class="container">© ' . $brand . '</div></footer>';
}

function get_pagination_data($totalItems, $perPage = 20, $pageParam = 'page') {
    $totalItems = max(0, (int)$totalItems);
    $perPage = max(1, (int)$perPage);

    $requestedPage = isset($_GET[$pageParam]) ? (int)$_GET[$pageParam] : 1;
    $currentPage = max(1, $requestedPage);
    $totalPages = max(1, (int)ceil($totalItems / $perPage));

    if ($currentPage > $totalPages) {
        $currentPage = $totalPages;
    }

    $offset = ($currentPage - 1) * $perPage;
    $fromItem = $totalItems === 0 ? 0 : $offset + 1;
    $toItem = min($totalItems, $offset + $perPage);

    return [
        'total_items' => $totalItems,
        'per_page' => $perPage,
        'current_page' => $currentPage,
        'total_pages' => $totalPages,
        'offset' => $offset,
        'has_prev' => $currentPage > 1,
        'has_next' => $currentPage < $totalPages,
        'from_item' => $fromItem,
        'to_item' => $toItem,
        'page_param' => $pageParam,
    ];
}

function render_admin_pagination($pagination) {
    if (empty($pagination) || ($pagination['total_pages'] ?? 1) <= 1) {
        return;
    }

    $currentPage = (int)$pagination['current_page'];
    $totalPages = (int)$pagination['total_pages'];
    $pageParam = $pagination['page_param'] ?? 'page';
    $query = $_GET;

    echo '<div class="pagination">';

    if ($pagination['has_prev']) {
        $query[$pageParam] = $currentPage - 1;
        echo '<a class="page-link" href="?' . htmlspecialchars(http_build_query($query), ENT_QUOTES, 'UTF-8') . '">&laquo; Zurück</a>';
    }

    for ($i = 1; $i <= $totalPages; $i++) {
        if ($i === 1 || $i === $totalPages || abs($i - $currentPage) <= 2) {
            $query[$pageParam] = $i;
            $activeClass = $i === $currentPage ? ' active' : '';
            echo '<a class="page-link' . $activeClass . '" href="?' . htmlspecialchars(http_build_query($query), ENT_QUOTES, 'UTF-8') . '">' . $i . '</a>';
        } elseif ($i === 2 || $i === $totalPages - 1) {
            echo '<span class="page-ellipsis">...</span>';
        }
    }

    if ($pagination['has_next']) {
        $query[$pageParam] = $currentPage + 1;
        echo '<a class="page-link" href="?' . htmlspecialchars(http_build_query($query), ENT_QUOTES, 'UTF-8') . '">Weiter &raquo;</a>';
    }

    echo '</div>';
}

function find_user_by_email($email) {
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM users WHERE email = ? LIMIT 1');
    $stmt->execute([$email]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function find_user_by_id($id) {
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM users WHERE id = ? LIMIT 1');
    $stmt->execute([$id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function login_user($user) {
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['email'] = $user['email'];
}

function logout_user() {
    session_unset();
    session_destroy();
}

function is_logged_in() {
    return !empty($_SESSION['user_id']);
}

function require_login() {
    if (!is_logged_in()) {
        header('Location: login.php');
        exit;
    }
}

function create_reset_token($userId) {
    $token = bin2hex(random_bytes(16));
    $expires = time() + 3600; // 1 hour
    $db = getDB();
    $stmt = $db->prepare('UPDATE users SET reset_token = ?, reset_expires = ? WHERE id = ?');
    $stmt->execute([$token, $expires, $userId]);
    return $token;
}

function find_user_by_token($token) {
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM users WHERE reset_token = ? LIMIT 1');
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) return null;
    if ($user['reset_expires'] < time()) return null;
    return $user;
}

function set_user_password($userId, $newPassword) {
    $hash = password_hash($newPassword, PASSWORD_DEFAULT);
    $db = getDB();
    $stmt = $db->prepare('UPDATE users SET password_hash = ?, reset_token = NULL, reset_expires = NULL WHERE id = ?');
    $stmt->execute([$hash, $userId]);
}

/* Email change workflow */
function create_email_token($userId, $newEmail) {
    $token = bin2hex(random_bytes(16));
    $expires = time() + 3600; // 1 hour
    $db = getDB();
    $stmt = $db->prepare('UPDATE users SET pending_email = ?, email_token = ?, email_expires = ? WHERE id = ?');
    $stmt->execute([$newEmail, $token, $expires, $userId]);
    return $token;
}

function find_user_by_email_token($token) {
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM users WHERE email_token = ? LIMIT 1');
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) return null;
    if ($user['email_expires'] < time()) return null;
    return $user;
}

function confirm_email_change($userId) {
    $db = getDB();
    $stmt = $db->prepare('SELECT pending_email FROM users WHERE id = ? LIMIT 1');
    $stmt->execute([$userId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row || empty($row['pending_email'])) return false;
    $stmt = $db->prepare('UPDATE users SET email = ?, pending_email = NULL, email_token = NULL, email_expires = NULL WHERE id = ?');
    $stmt->execute([$row['pending_email'], $userId]);
    return true;
}

/* Settings table helpers */
function get_setting($key, $default = null) {
    $db = getDB();
    $stmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1');
    $stmt->execute([$key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? $row['setting_value'] : $default;
}

function update_setting($key, $value) {
    $db = getDB();
    $stmt = $db->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?');
    return $stmt->execute([$key, $value, $value]);
}

/* Email sending with SMTP settings from database, returns detailed error info */
function send_mail_message($to, $subject, $body, $isHtml = true, &$errorMsg = null) {
    $errorMsg = null;
    
    // Create logs directory if not exists
    $logsDir = APP_ROOT . '/logs';
    if (!is_dir($logsDir)) @mkdir($logsDir, 0755, true);
    
    // Get SMTP settings from database
    $smtpHost = trim(get_setting('smtp_host', ''));
    $smtpPort = (int)(get_setting('smtp_port', '587'));
    $smtpUser = trim(get_setting('smtp_user', ''));
    $smtpPass = trim(get_setting('smtp_pass', ''));
    $smtpSecure = trim(get_setting('smtp_secure', ''));
    $fromEmail = trim(get_setting('from_email', 'no-reply@example.com'));
    $fromName = trim(get_setting('from_name', 'My Plugin'));
    
    // Log email attempt
    $logEntry = date('Y-m-d H:i:s') . " | TO: {$to} | SUBJECT: {$subject}\n";
    $logEntry .= "  SMTP_HOST: " . ($smtpHost ?: '[EMPTY - using mail()]') . " | SMTP_PORT: {$smtpPort}\n";
    $logEntry .= "  SMTP_USER: " . ($smtpUser ? '[SET]' : '[EMPTY]') . " | SMTP_SECURE: " . ($smtpSecure ?: '[NONE]') . "\n";
    @file_put_contents($logsDir . '/email.log', $logEntry, FILE_APPEND);

    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-type: " . ($isHtml ? 'text/html' : 'text/plain') . "; charset=utf-8\r\n";
    $headers .= "From: {$fromName} <{$fromEmail}>\r\n";
    $headers .= "Reply-To: {$fromEmail}\r\n";

    // If SMTP host not configured, use mail() function
    if (empty($smtpHost)) {
        $result = @mail($to, $subject, $body, $headers);
        @file_put_contents($logsDir . '/email.log', "  METHOD: PHP mail() | RESULT: " . ($result ? 'SUCCESS' : 'FAILED') . "\n\n", FILE_APPEND);
        if (!$result) {
            $errorMsg = 'Mail function failed. No SMTP configured and mail() failed.';
        }
        return $result;
    }

    // Try SMTP connection
    $socket = null;
    try {
        // Determine protocol and connect
        $protocol = ($smtpSecure === 'ssl' || $smtpSecure === 'SSL') ? 'ssl://' : '';
        $connectString = $protocol . $smtpHost;
        
        // Suppress warnings and connect with timeout
        $socket = @stream_socket_client($connectString . ':' . $smtpPort, $errno, $errstr, 30);
        
        if (!$socket) {
            $errorMsg = "SMTP connection failed to {$smtpHost}:{$smtpPort}. Error: {$errstr}";
            @file_put_contents($logsDir . '/email.log', "  METHOD: SMTP ({$smtpHost}:{$smtpPort}) | RESULT: CONNECTION FAILED | ERROR: {$errstr}\n", FILE_APPEND);
            // SMTP connection failed, use mail() fallback
            $result = @mail($to, $subject, $body, $headers);
            @file_put_contents($logsDir . '/email.log', "  FALLBACK: PHP mail() | RESULT: " . ($result ? 'SUCCESS' : 'FAILED') . "\n\n", FILE_APPEND);
            if ($result) {
                $errorMsg = null;
            } else {
                $errorMsg .= ' And mail() also failed as fallback.';
            }
            return $result;
        }

        // Function to safely read from socket (handles multi-line SMTP responses)
        $read_response = function($socket, $timeout = 5) {
            stream_set_timeout($socket, $timeout);
            $fullResponse = '';
            do {
                $response = @fgets($socket, 1024);
                if ($response === false) break;
                $fullResponse .= $response;
                // Multi-line responses use "XXX-" for continuation and "XXX " for final
                // Stop when we get a line starting with code followed by space (not dash)
                if (preg_match('/^\d{3} /', $response)) break;
            } while (true);
            return $fullResponse ?: '';
        };

        // Read server greeting
        $response = $read_response($socket);
        if (!$response || strpos($response, '220') === false) {
            $errorMsg = 'No greeting from SMTP server.';
            @fclose($socket);
            return @mail($to, $subject, $body, $headers);
        }

        // EHLO command
        @fwrite($socket, "EHLO localhost\r\n");
        $response = $read_response($socket);

        // STARTTLS if required
        if ($smtpSecure === 'tls' || $smtpSecure === 'TLS') {
            @fwrite($socket, "STARTTLS\r\n");
            $response = $read_response($socket);
            if (stripos($response, '220') !== false) {
                if (@stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                    @fwrite($socket, "EHLO localhost\r\n");
                    $response = $read_response($socket);
                } else {
                    $errorMsg = 'TLS/SSL negotiation failed.';
                    @fclose($socket);
                    return @mail($to, $subject, $body, $headers);
                }
            }
        }

        // Authentication if credentials provided
        if (!empty($smtpUser) && !empty($smtpPass)) {
            $authSuccess = false;
            $authAttempts = [];
            
            // Try AUTH LOGIN first
            @fwrite($socket, "AUTH LOGIN\r\n");
            $response = $read_response($socket);
            $authAttempts[] = "AUTH LOGIN response: " . trim($response);
            
            if (strpos($response, '334') !== false) {
                @fwrite($socket, base64_encode($smtpUser) . "\r\n");
                $response = $read_response($socket);
                @fwrite($socket, base64_encode($smtpPass) . "\r\n");
                $response = $read_response($socket);
                $authAttempts[] = "LOGIN complete response: " . trim($response);
                if (strpos($response, '235') !== false) {
                    $authSuccess = true;
                }
            }
            
            // If LOGIN failed, try AUTH PLAIN
            if (!$authSuccess) {
                $authString = base64_encode($smtpUser . ':' . $smtpPass);
                @fwrite($socket, "AUTH PLAIN {$authString}\r\n");
                $response = $read_response($socket);
                $authAttempts[] = "AUTH PLAIN (user:pass) response: " . trim($response);
                if (strpos($response, '235') !== false) {
                    $authSuccess = true;
                }
            }
            
            // If PLAIN also failed, try PLAIN with null byte format
            if (!$authSuccess) {
                $authString = base64_encode("\x00" . $smtpUser . "\x00" . $smtpPass);
                @fwrite($socket, "AUTH PLAIN {$authString}\r\n");
                $response = $read_response($socket);
                $authAttempts[] = "AUTH PLAIN (null byte) response: " . trim($response);
                if (strpos($response, '235') !== false) {
                    $authSuccess = true;
                }
            }
            
            // Try without sending AUTH if all methods fail (some servers don't require auth from localhost)
            if (!$authSuccess) {
                $authAttempts[] = "All AUTH methods failed. Attempting to send without authentication...";
                // Don't close socket, continue without auth
                // Reset for mail sending below
            }
            
            if (!$authSuccess) {
                $errorMsg = 'SMTP authentication failed. Details: ' . implode(' | ', $authAttempts);
                @fclose($socket);
                return @mail($to, $subject, $body, $headers);
            }
        }

        // MAIL FROM
        @fwrite($socket, "MAIL FROM:<{$fromEmail}>\r\n");
        $response = $read_response($socket);
        if (strpos($response, '250') === false) {
            $errorMsg = 'MAIL FROM command rejected by server.';
            @fclose($socket);
            return @mail($to, $subject, $body, $headers);
        }

        // RCPT TO
        @fwrite($socket, "RCPT TO:<{$to}>\r\n");
        $response = $read_response($socket);
        if (strpos($response, '250') === false && strpos($response, '251') === false) {
            $errorMsg = "Server rejected recipient address: {$to}";
            @fclose($socket);
            return @mail($to, $subject, $body, $headers);
        }

        // DATA
        @fwrite($socket, "DATA\r\n");
        $response = $read_response($socket);
        if (strpos($response, '354') === false) {
            $errorMsg = 'DATA command not accepted by server.';
            @fclose($socket);
            return @mail($to, $subject, $body, $headers);
        }

        // Compose and send message
        $message = "From: {$fromName} <{$fromEmail}>\r\n";
        $message .= "To: {$to}\r\n";
        $message .= "Subject: {$subject}\r\n";
        $message .= $headers . "\r\n";
        $message .= $body;

        @fwrite($socket, $message . "\r\n.\r\n");
        $response = $read_response($socket);
        if (strpos($response, '250') === false) {
            $errorMsg = 'Server rejected message. Response: ' . trim($response);
            @fclose($socket);
            return @mail($to, $subject, $body, $headers);
        }

        // QUIT
        @fwrite($socket, "QUIT\r\n");
        @fclose($socket);
        
        @file_put_contents($logsDir . '/email.log', "  METHOD: SMTP ({$smtpHost}:{$smtpPort}) | RESULT: SUCCESS\n\n", FILE_APPEND);
        return true;

    } catch (Exception $e) {
        if ($socket) @fclose($socket);
        $errorMsg = 'Exception: ' . $e->getMessage();
        @file_put_contents($logsDir . '/email.log', "  METHOD: SMTP ({$smtpHost}:{$smtpPort}) | RESULT: FAILED | ERROR: {$errorMsg}\n\n", FILE_APPEND);
        // Fallback to mail() on any exception
        return @mail($to, $subject, $body, $headers);
    }
}

/* Automatic email cron job - sends reports every 14 days */
function run_automatic_email_cron() {
    try {
        $db = getDB();
        $reportTable = table('WPAS_email_reports');
        
        // Check if a report was sent in the last 14 days
        $stmt = $db->prepare("
            SELECT id FROM " . $reportTable . "
            WHERE type = 'Automatic' AND date >= DATE_SUB(NOW(), INTERVAL 14 DAY)
            LIMIT 1
        ");
        $stmt->execute();
        $lastReport = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // If no recent automatic report, send one
        if (!$lastReport) {
            $adminEmail = get_setting('as_admin_email', '');
            
            if (empty($adminEmail)) {
                error_log('Cron: No admin email configured. Skipping automatic report.');
                return false;
            }
            
            // Get search records for the last 14 days
            $stmt = $db->prepare("
                SELECT * FROM " . table('WPAS_search_record') . "
                WHERE date >= DATE_SUB(NOW(), INTERVAL 14 DAY)
                ORDER BY date DESC
            ");
            $stmt->execute();
            $records = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Build email
            $tableRows = '';
            if (!empty($records)) {
                $i = 1;
                foreach ($records as $rec) {
                    $resultText = $rec['result'] == 1 ? 'Gefunden' : 'Nicht gefunden';
                    $tableRows .= "<tr><td>" . $i . "</td><td>" . htmlspecialchars($rec['username']) . "</td><td>" . htmlspecialchars($rec['certificate']) . "</td><td>" . $resultText . "</td><td>" . htmlspecialchars($rec['date']) . "</td></tr>";
                    $i++;
                }
            } else {
                $tableRows = '<tr><td colspan="5">Keine Einträge gefunden.</td></tr>';
            }
            
            $subject = 'Zusammenfassung der FfV Zertifikatsabfragen (Automatischer Report)';
            $body = '<html><body style="font-family: Space Grotesk, sans-serif;">';
            $body .= '<h2>Zertifikatsabfrage Report</h2>';
            $body .= '<p>Dies ist ein automatisch generierter Report der Zertifikatsabfragen der letzten 14 Tage.</p>';
            $body .= '<table rules="all" style="border: 1px solid #ccc; border-collapse: collapse; width: 100%;">';
            $body .= '<thead><tr style="background-color: #f5f5f5;"><th style="padding: 10px; border: 1px solid #ccc;">Nr.</th><th style="padding: 10px; border: 1px solid #ccc;">Lieferant</th><th style="padding: 10px; border: 1px solid #ccc;">Zertifikat</th><th style="padding: 10px; border: 1px solid #ccc;">Ergebnis</th><th style="padding: 10px; border: 1px solid #ccc;">Zeitstempel</th></tr></thead>';
            $body .= '<tbody>' . $tableRows . '</tbody>';
            $body .= '</table>';
            $body .= '<p style="margin-top: 20px; color: #999; font-size: 12px;">Dieser Report wurde automatisch am ' . date('d.m.Y H:i:s') . ' versendet. Der nächste Report wird in 14 Tagen versendet.</p>';
            $body .= '</body></html>';
            
            $emailError = null;
            $sent = send_mail_message($adminEmail, $subject, $body, true, $emailError);
            
            if ($sent) {
                // Record the report
                $stmt = $db->prepare("
                    INSERT INTO " . $reportTable . " (total_amount_requests, type) 
                    VALUES (?, ?)
                ");
                $stmt->execute([count($records), 'Automatic']);
                error_log('Cron: Automatic email report sent successfully. (' . count($records) . ' records)');
                return true;
            } else {
                error_log('Cron: Failed to send automatic email report. Error: ' . ($emailError ?? 'Unknown'));
                return false;
            }
        }
        
        return true; // Report already sent recently
        
    } catch (Exception $e) {
        error_log('Cron: Error running automatic email cron: ' . $e->getMessage());
        return false;
    }
}
