<?php
require_once __DIR__ . '/../includes/init.php';
require_login();

$user = find_user_by_id($_SESSION['user_id']);
$msg = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'plugin_settings') {
        $adminEmail = trim($_POST['admin_email'] ?? '');
        $maxRequests = trim($_POST['max_requests'] ?? '5');

        if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Please enter a valid admin email address.';
        } elseif (!is_numeric($maxRequests) || (int)$maxRequests < 1) {
            $error = 'Maximum requests per day must be at least 1.';
        } else {
            update_setting('as_admin_email', $adminEmail);
            update_setting('maximum_amount_request', (int)$maxRequests);
            $msg = 'Plugin settings saved successfully.';
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'send_manual_report') {
        $days = (int)($_POST['report_days'] ?? 7);

        if ($days < 1) {
            $error = 'Please select at least 1 day.';
        } else {
            try {
                $db = getDB();
                $adminEmail = get_setting('as_admin_email', 'admin@example.com');

                // Get search records for the specified period
                $stmt = $db->prepare("
                    SELECT * FROM " . table('WPAS_search_record') . "
                    WHERE date >= DATE_SUB(NOW(), INTERVAL ? DAY)
                    ORDER BY date DESC
                ");
                $stmt->execute([$days]);
                $records = $stmt->fetchAll(PDO::FETCH_ASSOC);

                // Build email
                $tableRows = '';
                if (!empty($records)) {
                    $i = 1;
                    foreach ($records as $rec) {
                        $resultText = $rec['result'] == 1 ? 'Found' : 'Not found';
                        $tableRows .= "<tr><td>" . $i . "</td><td>" . htmlspecialchars($rec['username']) . "</td><td>" . htmlspecialchars($rec['certificate']) . "</td><td>" . $resultText . "</td><td>" . htmlspecialchars($rec['date']) . "</td></tr>";
                        $i++;
                    }
                } else {
                    $tableRows = '<tr><td colspan="5">No records found.</td></tr>';
                }

                $subject = 'FFV Certificate Requests - Email Report';
                $body = '<html><body style="font-family: Space Grotesk, sans-serif;">';
                $body .= '<h2>Certificate Request Report</h2>';
                $body .= '<p>Summary of requests from the last ' . $days . ' days:</p>';
                $body .= '<table rules="all" style="border: 1px solid #ccc; border-collapse: collapse; width: 100%;">';
                $body .= '<thead><tr style="background-color: #f5f5f5;"><th style="padding: 10px; border: 1px solid #ccc;">No.</th><th style="padding: 10px; border: 1px solid #ccc;">Supplier</th><th style="padding: 10px; border: 1px solid #ccc;">Certificate</th><th style="padding: 10px; border: 1px solid #ccc;">Result</th><th style="padding: 10px; border: 1px solid #ccc;">Timestamp</th></tr></thead>';
                $body .= '<tbody>' . $tableRows . '</tbody>';
                $body .= '</table>';
                $body .= '<p style="margin-top: 20px; color: #999; font-size: 12px;">This report was sent manually on ' . date('d.m.Y H:i:s') . '.</p>';
                $body .= '</body></html>';

                $emailError = null;
                $sent = send_mail_message($adminEmail, $subject, $body, true, $emailError);

                if ($sent) {
                    // Record the report
                    $stmt = $db->prepare("
                        INSERT INTO " . table('WPAS_email_reports') . " (total_amount_requests, type)
                        VALUES (?, ?)
                    ");
                    $stmt->execute([count($records), 'Manual']);

                    $msg = 'Email report sent successfully to ' . htmlspecialchars($adminEmail) . ' (' . count($records) . ' records).';
                } else {
                    $error = 'Failed to send report: ' . ($emailError ?? 'Unknown error');
                }
            } catch (Exception $e) {
                $error = 'Error: ' . $e->getMessage();
            }
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'password') {
        $current = $_POST['current_password'] ?? '';
        $new = $_POST['new_password'] ?? '';
        $new2 = $_POST['new_password2'] ?? '';
        if (!password_verify($current, $user['password_hash'])) {
            $error = 'Current password is incorrect.';
        } elseif ($new !== $new2) {
            $error = 'New passwords do not match.';
        } elseif (strlen($new) < 6) {
            $error = 'New password too short (min 6 chars).';
        } else {
            set_user_password($user['id'], $new);
            $msg = 'Password updated successfully.';
            $user = find_user_by_id($user['id']);
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'smtp') {
        update_setting('smtp_host', trim($_POST['smtp_host'] ?? ''));
        update_setting('smtp_port', trim($_POST['smtp_port'] ?? '587'));
        update_setting('smtp_user', trim($_POST['smtp_user'] ?? ''));
        update_setting('smtp_pass', trim($_POST['smtp_pass'] ?? ''));
        update_setting('smtp_secure', trim($_POST['smtp_secure'] ?? ''));
        update_setting('from_email', trim($_POST['from_email'] ?? 'no-reply@example.com'));
        update_setting('from_name', trim($_POST['from_name'] ?? 'My Plugin'));
        $msg = 'SMTP settings saved successfully.';
    }

    if (isset($_POST['action']) && $_POST['action'] === 'test_email') {
        $testEmail = trim($_POST['test_email'] ?? '');
        if (!filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Please provide a valid email address for testing.';
        } else {
            $testSubject = 'Test Email from My Plugin';
            $testBody = '<h2>Test Email</h2><p>This is a test email to verify your SMTP configuration.</p><p>If you received this email, your SMTP settings are working correctly!</p>';
            $emailError = null;
            $sent = send_mail_message($testEmail, $testSubject, $testBody, true, $emailError);
            if ($sent) {
                $msg = "Test email sent successfully to {$testEmail}!";
            } else {
                if ($emailError) {
                    $error = 'Test email failed: ' . htmlspecialchars($emailError);
                } else {
                    $error = 'Test email failed. Check SMTP settings and try again.';
                }
            }
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'email') {
        $newEmail = trim($_POST['new_email'] ?? '');
        if (!filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Invalid email address.';
        } else {
            $token = create_email_token($user['id'], $newEmail);
            $confirmLink = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http')
                . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) . '/confirm_email.php?token=' . $token;
            $subject = 'Confirm your new email';
            $body = "<p>Please click the link below to confirm your new email address:</p><p><a href=\"{$confirmLink}\">{$confirmLink}</a></p>";
            $sent = send_mail_message($newEmail, $subject, $body, true);
            if ($sent) {
                $msg = 'Confirmation email sent to new address. Click the link to confirm.';
            } else {
                $error = 'Failed to send confirmation email. Please try again later.';
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Settings</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin-pages.css">
</head>
<body class="wp-admin admin-settings">
<?php render_admin_topbar($user['email']); ?>
<div class="container layout">
    <?php render_admin_sidebar('settings'); ?>
    <main class="main">
        <h1>Settings</h1>
        <?php if($error): ?><div class="notice error"><?php echo htmlspecialchars($error); ?></div><?php endif; ?>
        <?php if($msg): ?><div class="notice success"><?php echo htmlspecialchars($msg); ?></div><?php endif; ?>

        <div class="card">
            <h2>Change Password</h2>
            <form method="post">
                <input type="hidden" name="action" value="password">
                <label>Current password
                    <input type="password" name="current_password" required>
                </label>
                <label>New password
                    <input type="password" name="new_password" required>
                </label>
                <label>Confirm new password
                    <input type="password" name="new_password2" required>
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Change password</button>
                </div>
            </form>
        </div>

        <div class="card">
            <h2>Change Email</h2>
            <p>Current email: <strong><?php echo htmlspecialchars($user['email']); ?></strong></p>
            <form method="post">
                <input type="hidden" name="action" value="email">
                <label>New email address
                    <input type="email" name="new_email" required>
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Send confirmation</button>
                </div>
            </form>
        </div>

        <div class="card">
            <h2>Plugin Settings</h2>
            <p>Configure the core plugin settings.</p>
            <form method="post">
                <input type="hidden" name="action" value="plugin_settings">
                <label>Admin Email Address (for reports) *
                    <input type="email" name="admin_email" value="<?php echo htmlspecialchars(get_setting('as_admin_email', '')); ?>" required>
                </label>
                <label>Maximum Requests per Day (per browser session) *
                    <input type="number" name="max_requests" value="<?php echo htmlspecialchars(get_setting('maximum_amount_request', '5')); ?>" min="1" required>
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Save Settings</button>
                </div>
            </form>
        </div>

        <div class="card">
            <h2>SMTP Settings</h2>
            <p>Configure email sending. Leave empty to use PHP mail() instead.</p>
            <form method="post">
                <input type="hidden" name="action" value="smtp">
                <label>SMTP Host
                    <input type="text" name="smtp_host" value="<?php echo htmlspecialchars(get_setting('smtp_host', '')); ?>">
                </label>
                <label>SMTP Port
                    <input type="number" name="smtp_port" value="<?php echo htmlspecialchars(get_setting('smtp_port', '587')); ?>">
                </label>
                <label>SMTP User
                    <input type="text" name="smtp_user" value="<?php echo htmlspecialchars(get_setting('smtp_user', '')); ?>">
                </label>
                <label>SMTP Password
                    <input type="password" name="smtp_pass" value="<?php echo htmlspecialchars(get_setting('smtp_pass', '')); ?>">
                </label>
                <label>Security Type
                    <select name="smtp_secure">
                        <option value="" <?php echo get_setting('smtp_secure', '') === '' ? 'selected' : ''; ?>>None</option>
                        <option value="tls" <?php echo get_setting('smtp_secure') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                        <option value="ssl" <?php echo get_setting('smtp_secure') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                    </select>
                </label>
                <label>From Email
                    <input type="email" name="from_email" value="<?php echo htmlspecialchars(get_setting('from_email', 'no-reply@example.com')); ?>">
                </label>
                <label>From Name
                    <input type="text" name="from_name" value="<?php echo htmlspecialchars(get_setting('from_name', 'My Plugin')); ?>">
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Save SMTP Settings</button>
                </div>
            </form>
        </div>

        <div class="card">
            <h2>Test Email</h2>
            <p>Send a test email to verify your SMTP configuration.</p>
            <form method="post">
                <input type="hidden" name="action" value="test_email">
                <label>Test Email Address
                    <input type="email" name="test_email" required>
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Send Test Email</button>
                </div>
            </form>
        </div>

        <div class="card">
            <h2>Manual Email Report</h2>
            <p>Send an email report with all certificate requests from recent days.</p>
            <form method="post">
                <input type="hidden" name="action" value="send_manual_report">
                <label>Report for the last X days
                    <input type="number" name="report_days" value="7" min="1" max="365">
                </label>
                <div class="form-actions">
                    <button class="btn" type="submit">Send Report</button>
                </div>
            </form>
        </div>
    </main>
</div>

<?php render_admin_footer(); ?>

</body>
</html>
