<?php
require_once __DIR__ . '/../includes/init.php';
require_login();

$user = find_user_by_id($_SESSION['user_id']);
$msg = '';
$error = '';
$preview_data = null;
$import_count = 0;

// Handle CSV upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'import_csv') {
        if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
            $error = 'Bitte wählen Sie eine gültige CSV-Datei.';
        } else {
            $file = $_FILES['csv_file'];
            $fileName = $file['name'];
            
            // Validate file type
            if (!preg_match('/\.csv$/i', $fileName)) {
                $error = 'Nur CSV-Dateien sind erlaubt.';
            } else if ($file['size'] > 5 * 1024 * 1024) { // 5MB limit
                $error = 'Die Datei ist zu groß (max. 5MB).';
            } else {
                // Read CSV file
                $handle = fopen($file['tmp_name'], 'r');
                $csvData = [];
                $header = null;
                $lineNum = 0;
                $errors = [];

                while (($row = fgetcsv($handle, 1000, ',')) !== false) {
                    $lineNum++;
                    
                    // First row is header
                    if ($lineNum === 1) {
                        // Normalize header: lowercase, trim whitespace
                        $header = array_map('strtolower', array_map('trim', $row));
                        
                        // Check for required columns
                        if (!in_array('username', $header) || !in_array('certificate', $header)) {
                            $errors[] = 'CSV muss Spalten "username" und "certificate" enthalten.';
                            break;
                        }
                        continue;
                    }

                    // Validate data row
                    if (empty($row[0]) && empty($row[1]) && empty($row[2])) {
                        // Skip empty rows
                        continue;
                    }

                    $record = [];
                    foreach ($header as $idx => $colName) {
                        if (isset($row[$idx])) {
                            $record[$colName] = trim($row[$idx]);
                        }
                    }

                    // Validate required fields
                    if (empty($record['username']) || empty($record['certificate'])) {
                        $errors[] = "Zeile {$lineNum}: Name und Zertifikat sind erforderlich.";
                        continue;
                    }

                    $csvData[] = $record;
                }
                fclose($handle);

                if (!empty($errors)) {
                    $error = 'Fehler beim Parsen der CSV: ' . implode(' | ', $errors);
                } else if (empty($csvData)) {
                    $error = 'Die CSV-Datei enthält keine gültigen Daten.';
                } else {
                    // Store in session for preview
                    $_SESSION['csv_preview'] = $csvData;
                    $_SESSION['csv_filename'] = $fileName;
                    $preview_data = $csvData;
                }
            }
        }
    }

    // Handle actual import
    if (isset($_POST['action']) && $_POST['action'] === 'confirm_import') {
        if (empty($_SESSION['csv_preview'])) {
            $error = 'Keine Vorschaudaten gefunden. Bitte laden Sie die Datei erneut hoch.';
        } else {
            $csvData = $_SESSION['csv_preview'];
            
            try {
                $db = getDB();
                $tableName = table('WPAS_csv');
                
                // Prepare insert statement
                $stmt = $db->prepare("INSERT INTO " . $tableName . " (username, certificate, product) VALUES (?, ?, ?)");
                
                $importedCount = 0;
                $failedRows = [];
                
                foreach ($csvData as $idx => $row) {
                    try {
                        $username = $row['username'] ?? '';
                        $certificate = $row['certificate'] ?? '';
                        $product = $row['product'] ?? '';
                        
                        $stmt->execute([$username, $certificate, $product]);
                        $importedCount++;
                    } catch (Exception $e) {
                        $failedRows[] = ($idx + 2) . ': ' . $e->getMessage();
                    }
                }
                
                // Record the import
                $stmt2 = $db->prepare("INSERT INTO " . table('WPAS_csv_record') . " (total_entries) VALUES (?)");
                $stmt2->execute([$importedCount]);
                
                // Clear session
                unset($_SESSION['csv_preview']);
                unset($_SESSION['csv_filename']);
                
                $msg = "CSV erfolgreich importiert: {$importedCount} Einträge hinzugefügt.";
                if (!empty($failedRows)) {
                    $msg .= " Fehler: " . implode(' | ', array_slice($failedRows, 0, 3));
                }
                
                $import_count = $importedCount;
                
            } catch (Exception $e) {
                $error = 'Fehler beim Importieren: ' . $e->getMessage();
            }
        }
    }
}

// Use session preview if available and not just imported
if (empty($preview_data) && isset($_SESSION['csv_preview']) && empty($msg)) {
    $preview_data = $_SESSION['csv_preview'];
}
?>
<!doctype html>
<html lang="de">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>CSV Daten importieren</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin-pages.css">
</head>
<body class="wp-admin admin-csv-import">
<?php render_admin_topbar($user['email']); ?>
<div class="container layout">
    <?php render_admin_sidebar('csv_import'); ?>
    <main class="main">
        <h1>CSV Daten importieren</h1>
        
        <?php if($error): ?><div class="notice error"><?php echo htmlspecialchars($error); ?></div><?php endif; ?>
        <?php if($msg): ?><div class="notice success"><?php echo htmlspecialchars($msg); ?></div><?php endif; ?>

        <?php if (!$preview_data): ?>
            <!-- CSV Upload Form -->
            <div class="instructions">
                <h2>Anleitung zum Importieren von Lieferantendaten</h2>
                <ol>
                    <li>Bereiten Sie eine CSV-Datei mit Ihren Lieferantendaten vor</li>
                    <li>Die CSV muss folgende Spalten enthalten:
                        <ul style="margin: 10px 0 10px 40px;">
                            <li><strong>username</strong> - Name des Lieferanten</li>
                            <li><strong>certificate</strong> - Zertifikatsnummer</li>
                            <li><strong>product</strong> - Produkttyp (optional)</li>
                        </ul>
                    </li>
                    <li>Max. Dateigröße: 5MB</li>
                    <li>Nur .csv Daten​ werden akzeptiert</li>
                </ol>

                <h3 style="margin-top: 20px;">Beispiel CSV Format:</h3>
                <div class="sample-csv">username,certificate,product
Plusbaum Samen,CERT-001,Fichte
Müller GmbH,CERT-002,Kiefer
Schmidt AG,CERT-003,Eiche</div>
            </div>

            <div class="card">
                <h2>CSV-Datei hochladen</h2>
                <form method="post" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="import_csv">
                    
                    <div class="form-group">
                        <label for="csv_file">CSV-Datei auswählen *</label>
                        <input 
                            type="file" 
                            id="csv_file" 
                            name="csv_file" 
                            accept=".csv" 
                            required
                            style="padding: 10px; border: 2px solid #ddd; border-radius: 4px; width: 100%;"
                        >
                        <small style="color: #999;">Erlaubt: .csv Dateien bis 5MB</small>
                    </div>

                    <div class="form-actions">
                        <button class="btn" type="submit">Datei hochladen und Vorschau anzeigen</button>
                    </div>
                </form>
            </div>

        <?php else: ?>
            <!-- CSV Preview -->
            <div class="file-info">
                <strong>Vorschau:</strong> <?php echo htmlspecialchars($_SESSION['csv_filename']); ?> 
                (<?php echo count($preview_data); ?> Einträge zum Importieren)
            </div>

            <div class="card">
                <h2>Datenvorschau</h2>
                <p>Überprüfen Sie die folgenden Daten und bestätigen Sie den Import:</p>
                
                <div class="csv-preview">
                    <table class="csv-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Lieferant (Username)</th>
                                <th>Zertifikatsnummer</th>
                                <th>Produkttyp</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($preview_data as $idx => $row): ?>
                            <tr>
                                <td><?php echo $idx + 1; ?></td>
                                <td><?php echo htmlspecialchars($row['username'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($row['certificate'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($row['product'] ?? '-'); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="button-group">
                    <form method="post" style="flex: 1;">
                        <input type="hidden" name="action" value="confirm_import">
                        <button class="btn btn-success" type="submit" style="width: 100%;">✓ Import bestätigen</button>
                    </form>
                    
                    <form method="post" style="flex: 1;">
                        <input type="hidden" name="action" value="cancel_preview">
                        <button class="btn btn-secondary" type="submit" style="width: 100%;">✕ Abbrechen</button>
                    </form>
                </div>
            </div>

        <?php endif; ?>

        <?php if ($import_count > 0): ?>
        <div class="card" style="margin-top: 20px;">
            <h2>✓ Import abgeschlossen</h2>
            <p><strong><?php echo $import_count; ?></strong> Lieferantendaten wurden erfolgreich importiert.</p>
            <p>Sie können diese Daten jetzt auf der <a href="suppliers.php">Lieferanten-Seite</a> anschauen und bearbeiten.</p>
        </div>
        <?php endif; ?>

    </main>
</div>

<?php render_admin_footer(); ?>

</body>
</html>


